import Foundation
import SwiftUI

/// Service to manage medical disclaimer state and compliance tracking
/// Ensures App Store compliance and proper user consent management
class DisclaimerService: ObservableObject {
    @Published var hasAcceptedDisclaimer: Bool = false
    
    private let disclaimerKey = "hasAcceptedDisclaimer"
    private let disclaimerDateKey = "disclaimerAcceptedDate"
    private let disclaimerVersionKey = "disclaimerVersion"
    
    private let currentDisclaimerVersion = "2.0"
    
    init() {
        checkDisclaimerStatus()
    }
    
    /// Check if user has accepted the current version of disclaimer
    private func checkDisclaimerStatus() {
        let hasAccepted = UserDefaults.standard.bool(forKey: disclaimerKey)
        let acceptedVersion = UserDefaults.standard.string(forKey: disclaimerVersionKey) ?? "1.0"
        
        // Require re-acceptance if disclaimer version has been updated
        hasAcceptedDisclaimer = hasAccepted && acceptedVersion == currentDisclaimerVersion
    }
    
    /// Accept the medical disclaimer with full tracking
    func acceptDisclaimer() {
        let consentData: [String: Any] = [
            disclaimerKey: true,
            disclaimerDateKey: Date(),
            disclaimerVersionKey: currentDisclaimerVersion,
            "deviceModel": UIDevice.current.model,
            "systemVersion": UIDevice.current.systemVersion,
            "appVersion": Bundle.main.infoDictionary?["CFBundleShortVersionString"] as? String ?? "Unknown"
        ]
        
        // Store all consent data
        for (key, value) in consentData {
            UserDefaults.standard.set(value, forKey: key)
        }
        
        hasAcceptedDisclaimer = true
    }
    
    /// Reset disclaimer acceptance (for testing or privacy purposes)
    func resetDisclaimerAcceptance() {
        UserDefaults.standard.removeObject(forKey: disclaimerKey)
        UserDefaults.standard.removeObject(forKey: disclaimerDateKey)
        UserDefaults.standard.removeObject(forKey: disclaimerVersionKey)
        hasAcceptedDisclaimer = false
    }
    
    /// Get disclaimer acceptance information for privacy/audit purposes
    func getDisclaimerInfo() -> [String: Any] {
        return [
            "hasAccepted": UserDefaults.standard.bool(forKey: disclaimerKey),
            "acceptedDate": UserDefaults.standard.object(forKey: disclaimerDateKey) as? Date,
            "disclaimerVersion": UserDefaults.standard.string(forKey: disclaimerVersionKey) ?? "Unknown",
            "currentVersion": currentDisclaimerVersion,
            "needsReAcceptance": !hasAcceptedDisclaimer
        ]
    }
    
    /// Check if disclaimer needs to be shown
    var shouldShowDisclaimer: Bool {
        return !hasAcceptedDisclaimer
    }
}